/** @file   SpawnItems.cpp
 * @brief   Implementation of SpawnItem classes.
 * @version $Revision: 1.1 $
 * @date    $Date: 2006/03/16 21:38:25 $
 * @author  Tomi Lamminsaari
 */

#include "SpawnItems.h"
#include <sstream>
#include "ObjectFactory.h"
#include "WarGlobals.h"
#include "SpawnParameters.h"
#include "AnimPlayer.h"
#include "ElectricFence.h"
using std::istringstream;
using std::istream;
using std::string;
using std::vector;
using namespace eng2d;

namespace WeWantWar {


#define KOTLightFence   "<lightfence>"
#define KCTLightFence   "</lightfence>"
#define KOTParticles    "<particles>"
#define KCTParticles    "</particles>"
#define KOTAnimation    "<animation>"
#define KCTAnimation    "</animation>"
#define KOTSound        "<atmospheric_sound>"
#define KCTSound        "</atmospheric_sound>"
#define KOTBonus        "<bonusitem>"
#define KCTBonus        "</bonusitem>"
#define KOTTeleporter   "<teleporter>"
#define KCTTeleporter   "</teleporter>"
#define KGameObjectTags "<alien><corpse><sentrygun><car><civilian><fighter>"\
                        "<tank><mine><npc_character><decorative><barrel>"\
                        "<stargate><machinegun>"


GameObjectSpawnItem::GameObjectSpawnItem( const string& aObjectTag ) :
  iOpeningTag( aObjectTag ),
  iClosingTag( aObjectTag )
{
  string::iterator it = iClosingTag.begin() + 1;
  iClosingTag.insert( it, '/' );
}

GameObjectSpawnItem::~GameObjectSpawnItem()
{
}

void GameObjectSpawnItem::Launch()
{
  GameObject* object = ObjectFactory::CreateGameObject( this );
  if ( object != 0 ) {
    WarGlobals::pObjTable->addObject( object );
  }
}

string GameObjectSpawnItem::GetLiteral( LiteralId aId ) const
{
  string literal = "";
  switch ( aId ) {
    case ( EClosingTag ): {
      literal = iClosingTag;
      break;
    }
    case ( EOpeningTag ): {
      literal = iOpeningTag;
      break;
    }
    default: {
      break;
    }
  }
  return literal;
}



///
/// AnimationSpawnItem class
/// ========================
AnimationSpawnItem::AnimationSpawnItem() :
  SpawnItem()
{
}

AnimationSpawnItem::~AnimationSpawnItem()
{
}

void AnimationSpawnItem::Launch()
{
  Vec2D animPos( iParameters.getFloat(KParamNamePosX),
                 iParameters.getFloat(KParamNamePosY) );
  animPos *= 32;
  animPos += Vec2D( iParameters.getFloat(KParamNameOffsetX),
                    iParameters.getFloat(KParamNameOffsetY) );
  Vec2D movement( iParameters.getFloat(KParamNameAnimMoveX),
                  iParameters.getFloat(KParamNameAnimMoveY) );
  int animUid = iParameters.getInt(KParamNameUid);
  const Animation& anim = GameAnims::findAnimation( animUid );
  AnimPlayer::spawn( anim, animPos, movement, 0 );
}

string AnimationSpawnItem::GetLiteral( LiteralId aId ) const
{
  string literal = "";
  switch ( aId ) {
    case ( EClosingTag ): {
      literal = KCTAnimation;
      break;
    }
    case ( EOpeningTag ): {
      literal = KOTAnimation;
      break;
    }
    default: {
      break;
    }
  }
  return literal;
}

int AnimationSpawnItem::ReadParameter(istream& aIn, const string& aParameter)
{
  if ( aParameter == "dir:" ) {
    string dirX, dirY;
    aIn >> dirX >> dirY;
    iParameters.addParameter( KParamNameAnimMoveX, dirX );
    iParameters.addParameter( KParamNameAnimMoveY, dirY );
    
  } else {
    return SpawnItem::ReadParameter(aIn, aParameter);
  }
  return KErrNone;
}



///
/// BonusObjectSpawnItem class
/// ==========================

BonusObjectSpawnItem::BonusObjectSpawnItem() :
  SpawnItem()
{
}

BonusObjectSpawnItem::~BonusObjectSpawnItem()
{
}

void BonusObjectSpawnItem::Launch()
{
  Vec2D pos( iParameters.getFloat(KParamNamePosX),
             iParameters.getFloat(KParamNamePosY) );
  pos *= 32;
  pos += Vec2D( iParameters.getFloat(KParamNameOffsetX),
                iParameters.getFloat(KParamNameOffsetY) );
  string bonusName = iParameters.get(KParamNameSubCategory);
  BonusObject::Type bonusType = BonusObject::stringToType(bonusName);
  
  BonusObject* bonus = new BonusObject( bonusType );
  bonus->position( pos );
  WarGlobals::addBonusObject( bonus );
}

string BonusObjectSpawnItem::GetLiteral( LiteralId aId ) const
{
  string literal = "";
  switch ( aId ) {
    case ( EClosingTag ): {
      literal = KCTBonus;
      break;
    }
    case ( EOpeningTag ): {
      literal = KOTBonus;
      break;
    }
    default: {
      break;
    }
  }
  return literal;
}



///
/// SoundSpawnItem class
/// ====================

SoundSpawnItem::SoundSpawnItem() :
  SpawnItem()
{
}

SoundSpawnItem::~SoundSpawnItem()
{
}

void SoundSpawnItem::Launch()
{
}

string SoundSpawnItem::GetLiteral( LiteralId aId ) const
{
  string literal = "";
  switch ( aId ) {
    case ( EClosingTag ): {
      literal = KCTSound;
      break;
    }
    case ( EOpeningTag ): {
      literal = KOTSound;
      break;
    }
    default: {
      break;
    }
  }
  return literal;
}



///
/// ParticleSpawnItem class
/// =======================

ParticleSpawnItem::ParticleSpawnItem() :
  SpawnItem()
{
}

ParticleSpawnItem::~ParticleSpawnItem()
{
}

void ParticleSpawnItem::Launch()
{
  Vec2D pos( iParameters.getFloat(KParamNamePosX),
             iParameters.getFloat(KParamNamePosY) );
  pos *= 32;
  pos += Vec2D( iParameters.getFloat(KParamNameOffsetX),
                iParameters.getFloat(KParamNameOffsetY) );
  Vec2D dirVec(0,-10);
  dirVec.rotate( iParameters.getFloat(KParamNameAngle) );
  int particleCount = iParameters.getInt( KParamNameParticleCount );
  if ( particleCount < 1 ) {
    particleCount = 10;
  }
  
  // Check the subcategory of the particles we should spawn and create
  // right type of particle system for it.
  string subType = iParameters.get(KParamNameSubCategory);
  ParticleSystem* partSystem = 0;
  if ( subType == "glass" ) {
    partSystem = new ParticleGlass( pos, dirVec, particleCount );
  } else if ( subType == "sparks" ) {
    partSystem = new ParticleSparks( pos, dirVec, particleCount );
  } else if ( subType == "blood" ) {
    Color c( 140,80,40 );
    partSystem = new ParticleBlood( pos, dirVec, particleCount, c );
  }
  if ( partSystem != 0 ) {
    WarGlobals::pPartManager->addSystem( partSystem );
  }
}

string ParticleSpawnItem::GetLiteral( LiteralId aId ) const
{
  string literal = "";
  switch ( aId ) {
    case ( EClosingTag ): {
      literal = KCTParticles;
      break;
    }
    case ( EOpeningTag ): {
      literal = KOTParticles;
      break;
    }
    default: {
      break;
    }
  }
  return literal;
}

int ParticleSpawnItem::ReadParameter( istream& aIn, const string& aParameter)
{
  string tmp;
  if ( aParameter == "num:" ) {
    aIn >> tmp;
    iParameters.addParameter( KParamNameParticleCount, tmp );
    
  } else {
    return SpawnItem::ReadParameter(aIn, aParameter);
  }
  return KErrNone;
}



///
/// LightFenceSpawnItem class
/// =========================

LightFenceSpawnItem::LightFenceSpawnItem() :
  SpawnItem()
{
}

LightFenceSpawnItem::~LightFenceSpawnItem()
{
}

void LightFenceSpawnItem::Launch()
{
  Vec2D nodePos1( iParameters.getFloat(KParamNameLightFenceNode1X),
                  iParameters.getFloat(KParamNameLightFenceNode1Y) );
  Vec2D nodePos2( iParameters.getFloat(KParamNameLightFenceNode2X),
                  iParameters.getFloat(KParamNameLightFenceNode2Y) );
  nodePos1 *= 32;
  nodePos2 *= 32;
  nodePos1 += Vec2D( iParameters.getFloat(KParamNameLightFenceNode1OffsetX),
                     iParameters.getFloat(KParamNameLightFenceNode1OffsetY) );
  nodePos2 += Vec2D( iParameters.getFloat(KParamNameLightFenceNode2OffsetX),
                     iParameters.getFloat(KParamNameLightFenceNode2OffsetY) );

  // Create the fence object.
  LightFence* fence = 0;
  string subType = iParameters.get(KParamNameSubCategory);
  if ( subType == "electric" ) {
    fence = new ElectricFence( nodePos1, nodePos2 );
  }
  
  // Set the sequence.
  if ( fence != 0 ) {
    string seqString = iParameters.get(KParamNameLightFenceSequence);
    if ( seqString.length() > 0 ) {
      vector<LightFence::StateProp> stateSequence;
      istringstream iss(seqString);
      while ( iss.eof() != true ) {
        LightFence::StateProp s;
        iss >> s.state >> s.duration;
        stateSequence.push_back( s );
      }
      fence->setStateSequence( stateSequence );
    }
    
    WarGlobals::pFenceManager->addLightFence(fence);
  }
}

string LightFenceSpawnItem::GetLiteral( LiteralId aId ) const
{
  string literal = "";
  switch ( aId ) {
    case ( EClosingTag ): {
      literal = KCTLightFence;
      break;
    }
    case ( EOpeningTag ): {
      literal = KOTLightFence;
      break;
    }
    default: {
      break;
    }
  }
  return literal;
}

int LightFenceSpawnItem::ReadParameter(istream& aIn, const string& aParameter)
{
  if ( aParameter == "node1:" ) {
    string posX, posY, offsetX, offsetY;
    aIn >> posX >> posY >> offsetX >> offsetY;
    iParameters.addParameter( KParamNameLightFenceNode1X, posX );
    iParameters.addParameter( KParamNameLightFenceNode1Y, posY );
    iParameters.addParameter( KParamNameLightFenceNode1OffsetX, offsetX );
    iParameters.addParameter( KParamNameLightFenceNode1OffsetY, offsetY );
    
  } else if ( aParameter == "node2:" ) {
    string posX, posY, offsetX, offsetY;
    aIn >> posX >> posY >> offsetX >> offsetY;
    iParameters.addParameter( KParamNameLightFenceNode2X, posX );
    iParameters.addParameter( KParamNameLightFenceNode2Y, posY );
    iParameters.addParameter( KParamNameLightFenceNode2OffsetX, offsetX );
    iParameters.addParameter( KParamNameLightFenceNode2OffsetY, offsetY );
    
  } else if ( aParameter == "sequence:" ) {
    string sequenceString = "";
    while ( true ) {
      if ( aIn.eof() == true ) {
        return KErrEof;
      }
      string tmp;
      aIn >> tmp;
      if ( tmp == "*" ) {
        break;
        
      } else {
        sequenceString += tmp;
        sequenceString += " ";
      }
    }
    iParameters.addParameter( KParamNameLightFenceSequence, sequenceString );
      
  } else {
    return SpawnItem::ReadParameter(aIn, aParameter);
    
  }
  return KErrNone;
}



///
/// TeleporterSpawnItem class
/// =========================

TeleporterSpawnItem::TeleporterSpawnItem() :
  SpawnItem()
{
}

TeleporterSpawnItem::~TeleporterSpawnItem()
{
}

void TeleporterSpawnItem::Launch()
{
  Teleporter* teleporter = new Teleporter;
  if ( teleporter != 0 ) {
    teleporter->m_targetPos = Vec2D(iParameters.getFloat(KParamNameTargetX),
                                    iParameters.getFloat(KParamNameTargetY));
    teleporter->m_targetPos *= 32;
    teleporter->m_targetAngle = iParameters.getInt(KParamNameAngle);
    teleporter->m_rect.topleft = Vec2D(iParameters.getFloat(KParamNameTeleportTlX),
                                       iParameters.getFloat(KParamNameTeleportTlY));
    teleporter->m_rect.topleft *= 32;
    teleporter->m_rect.bottomright = Vec2D(iParameters.getFloat(KParamNameTeleportBrX),
                                           iParameters.getFloat(KParamNameTeleportBrY));
    teleporter->m_rect.bottomright *= 32;
    teleporter->m_id = iParameters.getInt(KParamNameIdCode);
    WarGlobals::teleporterList.push_back( teleporter );
  }
}

string TeleporterSpawnItem::GetLiteral( LiteralId aId ) const
{
  string literal = "";
  switch ( aId ) {
    case ( EClosingTag ): {
      literal = KCTTeleporter;
      break;
    }
    case ( EOpeningTag ): {
      literal = KOTTeleporter;
      break;
    }
    default: {
      break;
    }
  }
  return literal;
}

int TeleporterSpawnItem::ReadParameter(istream& aIn, const string& aParameter)
{
  if ( aParameter == "rect:" ) {
    string tlX, tlY, brX, brY;
    aIn >> tlX >> tlY >> brX >> brY;
    iParameters.addParameter( KParamNameTeleportTlX, tlX );
    iParameters.addParameter( KParamNameTeleportTlY, tlY );
    iParameters.addParameter( KParamNameTeleportBrX, brX );
    iParameters.addParameter( KParamNameTeleportBrY, brY );
    
  } else if ( aParameter == "dest:" ) {
    string posX, posY, angle;
    aIn >> posX >> posY >> angle;
    iParameters.addParameter(KParamNameTargetX, posX);
    iParameters.addParameter(KParamNameTargetY, posY);
    iParameters.addParameter(KParamNameAngle, angle);
    
  } else {
    return SpawnItem::ReadParameter(aIn, aParameter);
  }
  return KErrNone;
}


///
/// SpawnItemFactory
/// ================

SpawnItem* SpawnItemFactory::CreateSpawnItem( const string& aTag )
{
  SpawnItem* item = 0;
  string gameObjectTags( KGameObjectTags );
  
  if ( gameObjectTags.find(aTag) != string::npos ) {
    item = new GameObjectSpawnItem(aTag);
    
  } else if ( aTag == KOTBonus ) {
    item = new BonusObjectSpawnItem;
    
  } else if ( aTag == KOTAnimation ) {
    item = new AnimationSpawnItem;
    
  } else if ( aTag == KOTSound ) {
    item = new SoundSpawnItem;
    
  } else if ( aTag == KOTParticles ) {
    item = new ParticleSpawnItem;
    
  } else if ( aTag == KOTLightFence ) {
    item = new LightFenceSpawnItem;
    
  } else if ( aTag == KOTTeleporter ) {
    item = new TeleporterSpawnItem;
    
  } else {
    item = 0;
    
  }
  return item;
}

};  // end of namespace
